* racd12.do  January 2013 for Stata version 12

capture log close
log using racd12.txt, text replace

********** OVERVIEW OF racd12.do **********

* STATA Program 
* copyright C 2013 by A. Colin Cameron and Pravin K. Trivedi 
* used for "Regression Analyis of Count Data" SECOND EDITION
* by A. Colin Cameron and Pravin K. Trivedi (2013)
* Cambridge University Press

* This STATA program does Bayesian MCMC
*   12.4 MARKOV CHAIN MONTE CARLO METHODS

* To run you need no data (the data are generated)
* in your directory

********** SETUP **********

set more off
version 12
clear all
* set linesize 82
set scheme s1mono  /* Graphics scheme */
* set maxvar 100 width 1000

************

* Bayesian Random Walk Metropolis-Hastings with uniformative prior
* for Poisson regression using generated data
* Based on robit example of Koop (2003) chapter 9.3 "Bayesian Econometrics"
* Koop's Matlab program ch9artdatb.m rewritten for Poisson not probit
* plus adaptation to use random walk chain MH rather than data augmentation

********** SETUP **********

set mem 5m
set more off
set scheme s1mono
* version 11

********** CREATE DATA AND SUMMARIZE **********

* Generate artificial data set for Poisson
*  - explanatory variable x ~ N[0,1]sum
*  - dependent variable   y = Poisson(exp(-0.5*x) 
*                     for x ~ N[0,1]

set obs 100
set seed 10101
gen x = rnormal()
gen mu = exp(-0.5*x)
gen y = rpoisson(mu)
gen cons = 1
sum

* Alternative seed
* set seed 12345

********** POISSON MAXIMUM LIKELIHOOD ESTIMATION (for Table 12.1)

poisson y x
estimates store POISSON

********** POISSON BAYESIAN WITH RANDOM WALK CHAIN METROPOLIS HASTNGS **********

* Globals for number of reps and a key tuning parameter
 
global s1 10000      // number of retained reps
global s0 10000      // number of burnin reps
global sdscale 0.25  // use random walk b + $sdscale * N(0, I)

* Mata to obtain the posterior draws of b

mata
  // Create y vector and X matrix from Stat data set using st_view command
  st_view(y=., ., "y")            // dependent
  st_view(X=., ., ("cons", "x"))  // regressors
  Xnames = ("cons", "x")          // used to label output

  // Calculate a few quantities outside the loop for later use
  n = rows(X)
  k = cols(X)
  ones = J(n,1,1)

  // Specify the number of replications 
  s0 = $s0     // number of burnin reps
  s1 = $s1     // number of retained reps
  s = s0+s1    // total reps

  // Store all draws and MH acceptance ratein the following matrices
  b_all = J(k,s1,0)
  accept_all = J(1,s1,0) 

  // Initialization
  bdraw = J(k,1,0)      // starting b value is vector of zeroes
  lpostdraw = -1*10^10  // starting value of ln(posterior) is small 
                        // so accept initial MH draw
   
  // Now do Metropolis-Hastings loop 

  for (irep=1; irep<=s; irep++) {

     // Draw new candidate value of b from MH random walk chain
     bcandidate = bdraw + $sdscale*rnormal(k,1,0,1)

     // Note: For different data you may need to change the global sdscale
     // And best is bcandidate = bdraw + z  where z ~ N(0, posterior variance of b)
 
     // Compute the log posterior at the candidate value of b
     // The assumed prior for b is uninformative 
     // so the posterior is proportional to the usual Poisson likelihood
     // and for comparsions of lnL we can drop term ln(y_i!)
     Xb = X*bcandidate
     lpostcandidate = ones'(-exp(Xb) + y:*Xb)
 
     // Accept the candidate draw on basis of posterior probability ratio 
     // if  uniform > (posterior(bcandidate) / posterior(bdraw))
     // where bcandidate is current b  and  bdraw is previous b
     // Taking logs the rule is the same as
     // if  ln(uniform) > (lpostcandidate - lpostdraw)
     laccprobability = lpostcandidate - lpostdraw
     accept = 0
     if ( ln(runiform(1,1)) < laccprobability ) {
       lpostdraw = lpostcandidate
       bdraw = bcandidate    
       accept = 1 
       }

    // Store the draws after burn-in of b and whether accept draw 
    if (irep>s0) {
        // after discarding burnin, store all draws
        j = irep-s0
        b_all[.,j] = bdraw         // These are the posterior draws
        accept_all[.,j] = accept   // These are one if new draw accepted
    } 

  }                
  
  // End MH loop 

  // Pass results back to Stata
  // This bit works only for k = 2 (intercept plus one slope)
  beta = b_all'
  accept = accept_all'
  st_addvar("float", ("beta1", "beta2", "accept"))
  // The following line is needed for conformability 
  // It assumes that s1 (number of draws) exceeds the original sample size 
  stata("set obs $s1")
  st_store(., ("beta1", "beta2"), beta)  
  st_store(., ("accept"), accept)  
end

********** ANALYZE THE RESULTS **********

*** TABLE 12.1: BAYESIAN POSTERIOR SUMMARY 

* Posterior means and standard deviations
summarize beta1 beta2    

* Posterior credible regions
centile beta1 beta2, centile(2.5 97.5)

summarize
* save bayespoisson.dta, replace
rename beta2 b

* MLE of Poisson
estimates table POISSON, b(%7.4f) se(%7.3f) stats(N ll) stfmt(%9.1f)

*** FIGURE 12.1: POSTERIOR DRAWS AND ACF

* Plot the posterior draws of b2 and acf
generate s= _n
tsset s
line b s if s < 100, name(first100, replace) ytitle("{&beta}{sub:2}") ///
  xtitle("Draw s") scale(1.9)
line b s if s < 2000, name(all, replace) ytitle("{&beta}{sub:2}")     ///
  xtitle("Draw s") scale(1.9)
graph combine first100 all, iscale(1.0) ysize(3) xsize(6) ycommon
graph export racd12fig1.wmf, replace
graph export racd12fig1.eps, replace

*** FIGURE 12.2: POSTERIOR DENSITY

kdensity b, note("") legend(position(0)) xtitle("{&beta}{sub:2}") ///
   ytitle("Posterior density") title("") scale(1.9)
graph export racd12fig2.wmf, replace
graph export racd12fig2.eps, replace

*** OTHER DETAILS IN TEXT

* Correlation of the draws
corrgram b, lags(20)

* Give the acceptance rate in the random walk chain MH 
quietly summarize accept
display "MH acceptance rate = " r(mean) "

* Check the efficiency loss due to correlation of draws
regress b
newey b, lag(40)

********** CLOSE OUTPUT **********

* log close
* clear 
* exit
